﻿using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.GUI;
using gov.va.med.vbecs.GUI.WPF;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows;
using WPF.PresentationLayer.Resources;
using WPF.PresentationLayer.UC115.Events;
using WPF.PresentationLayer.UC115.Views;

namespace WPF.PresentationLayer.UC115.ViewModels
{
    public class PatientSelectionViewModel : ViewModelBase
    {
        #region Properties
        public DelegateCommand OKCommand { get; private set; }
        public DelegateCommand CancelCommand { get; private set; }
        public DelegateCommand SearchCommand { get; private set; }
        public DelegateCommand ClearCommand { get; private set; }

        private Dictionary<Guid, FrmWPFContainer> _openSpecimenFormDictionary;

        private PatientSelectionItemViewModel _selectedPatient;
        public PatientSelectionItemViewModel SelectedPatient
        {
            get
            {
                return _selectedPatient;
            }
            set
            {
                _selectedPatient = value;

                if (_selectedPatient == null)
                {
                    PatientInfoBarViewModel = null;
                }
                else
                {
                    PatientInfoBarViewModel = new PatientInfoBarViewModel(_selectedPatient.PatientWithSpecimen.Patient, VbecsBaseForm);
                }

                RaisePropertyChanged(() => SelectedPatient);
                RaisePropertyChanged(() => IsPatientSelected);
                RaisePropertyChanged(() => PatientLocationToolTip);
                RaisePropertyChanged(() => AboRhToolTip);
                RaisePropertyChanged(() => SelectedPatientLocationName);
            }
        }

        /// <summary>
        /// Patient Location Name
        /// </summary>
        public string SelectedPatientLocationName
        {
            get
            {
                if (SelectedPatient == null)
                {
                    return "";
                }
                else
                {
                    return PatientLocation.GetName(GetPatientLocation(SelectedPatient.PatientGuid));
                }
            }
        }

        private PatientInfoBarViewModel _patientInfoBarViewModel;
        /// <summary>
        /// Patient Info Bar View Model
        /// </summary>
        public PatientInfoBarViewModel PatientInfoBarViewModel
        {
            get
            {
                return _patientInfoBarViewModel;
            }
            set
            {
                _patientInfoBarViewModel = value;
                RaisePropertyChanged(() => PatientInfoBarViewModel);
            }
        }

        /// <summary>
        /// IsPatientSelected
        /// </summary>
        public bool IsPatientSelected
        {
            get
            {
                return (SelectedPatient != null);
            }
        }

        /// <summary>
        /// Full list of patients used to filter/search for patients and reset the list
        /// </summary>
        public ObservableCollection<PatientSelectionItemViewModel> PatientFullList { get; private set; }

        private ObservableCollection<PatientSelectionItemViewModel> _patientList;
        public ObservableCollection<PatientSelectionItemViewModel> PatientList
        {
            get
            {
                return _patientList;
            }
            set
            {
                _patientList = value;
                RaisePropertyChanged(() => PatientList);
            }
        }

        private string _specimenUidSearch;
        public string SpecimenUidSearch
        {
            get
            {
                return _specimenUidSearch;
            }
            set
            {
                _specimenUidSearch = value;
                RaisePropertyChanged(() => SpecimenUidSearch);
                RaisePropertyChanged(() => IsSearchEnabled);
            }
        }

        private string _patientSearch;
        public string PatientSearch
        {
            get
            {
                return _patientSearch;
            }
            set
            {
                _patientSearch = value;
                RaisePropertyChanged(() => PatientSearch);
                RaisePropertyChanged(() => IsSearchEnabled);
            }
        }

        public bool IsSearchEnabled
        {
            get
            {
                return !string.IsNullOrWhiteSpace(SpecimenUidSearch) || !string.IsNullOrWhiteSpace(PatientSearch);
            }
        }

        private IDictionary<Guid, PatientLocation> _patientLocationDictionary;

        #region ToolTips
        /// <summary>
        /// NameToolTip
        /// </summary>
        public string NameToolTip { get { return StrRes.Tooltips.Patient.NameToolTip().ResString; } }
        /// <summary>
        /// SsnToolTip
        /// </summary>
        public string SsnToolTip { get { return StrRes.Tooltips.Patient.SsnToolTip().ResString; } }
        /// <summary>
        /// DobToolTip
        /// </summary>
        public string DobToolTip { get { return StrRes.Tooltips.Patient.DateOfBirthToolTip().ResString; } }
        /// <summary>
        /// SexToolTip
        /// </summary>
        public string SexToolTip { get { return StrRes.Tooltips.Patient.SexToolTip().ResString; } }
        /// <summary>
        /// AboRhToolTip
        /// </summary>
        public string AboRhToolTip
        {
            get
            {
                var toolTip = StrRes.Tooltips.Patient.AboRhToolTip().ResString;
                if (SelectedPatient != null)
                {
                    toolTip += "\n" + SelectedPatient.PatientWithSpecimen.Patient.GetPatientHistoricABORhForTooltip();
                }

                return toolTip;
            }
        }
        /// <summary>
        /// Patient Location Name
        /// </summary>
        public string PatientLocationToolTip
        {
            get
            {
                if (SelectedPatient == null)
                {
                    return StrRes.Tooltips.Patient.LocationToolTip().ResString; ;
                }
                else
                {
                    return PatientLocation.GetToolTip(GetPatientLocation(SelectedPatient.PatientGuid));
                }
            }
        }
        #endregion
        #endregion

        public PatientSelectionViewModel()
        {
            LogoHeaderText = "Automated Testing - Pending Patient Orders";
            FormText = "VBECS - " + LogoHeaderText;

            PropertyChanged += PatientSelectionViewModel_PropertyChanged;

            _openSpecimenFormDictionary = new Dictionary<Guid, FrmWPFContainer>();
            _patientLocationDictionary = new Dictionary<Guid, PatientLocation>();

            OKCommand = new DelegateCommand(OnOKCommand);
            CancelCommand = new DelegateCommand(OnCancelCommand);
            SearchCommand = new DelegateCommand(OnSearchCommand);
            ClearCommand = new DelegateCommand(OnClearCommand);

            Refresh();

            Messenger.Default.Register<PendingTestReviewedEvent>(this, OnPendingTestReviewedEvent);

            LockFunction = LockFunctions.UC115AutomatedTestingReview;
        }

        /// <summary>
        /// Clean up events after the form closes
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void VbecsBaseForm_FormClosed(object sender, System.Windows.Forms.FormClosedEventArgs e)
        {
            if (VbecsBaseForm != null)
            {
                Messenger.Default.Unregister(this);
                PropertyChanged -= PatientSelectionViewModel_PropertyChanged;
                VbecsBaseForm.FormClosed -= VbecsBaseForm_FormClosed;
            }
        }

        private PatientLocation GetPatientLocation(Guid patientGuid)
        {
            // Cache patient location for each unique patient
            PatientLocation patientLocation;
            if (!_patientLocationDictionary.TryGetValue(patientGuid, out patientLocation))
            {
                patientLocation = PatientLocation.GetCurrentPatientLocation(patientGuid);
                _patientLocationDictionary.Add(patientGuid, patientLocation);
            }

            return patientLocation;
        }

        private void Refresh()
        {
            // Hold onto the currently selected specimenUid so we can re-select it
            var selectedSpecimenUid = SelectedPatient == null ? "" : SelectedPatient.PatientWithSpecimen.SpecimenUid;

            var patientDataList = PendingSpecimenTest.GetPatientsWithPendingSpecimenTests(LogonUser.LogonUserDivisionCode, PendingTestStatus.Pending).ToObservableCollection();

            if (patientDataList != null)
            {
                PatientList = patientDataList.OrderBy(x => x.SpecimenUid)
                                             .Select(s => new PatientSelectionItemViewModel(s))
                                             .ToObservableCollection();
                PatientFullList = PatientList;
            }

            SearchCommand.Execute();

            if (!string.IsNullOrWhiteSpace(selectedSpecimenUid))
            {
                // Re-select the previously selected specimen uid if it exists
                SelectedPatient = PatientList.FirstOrDefault(x => x.PatientWithSpecimen.SpecimenUid == selectedSpecimenUid);
            }

            if (!PatientFullList.Any() &&
                VbecsBaseForm != null)
            {
                // No results exist any more so lets close the form
                VbecsBaseForm.Close();
            }
        }

        private void OnPendingTestReviewedEvent(PendingTestReviewedEvent payload)
        {
            if (payload != null &&
                payload.PendingTestList != null)
            {
                // We only care about IPendingSpecimenTestModel tests
                if (payload.PendingTestList.Any(s => s is IPendingSpecimenTestModel))
                {
                    Refresh();
                }
                //var pendingSpecimenList = payload.PendingTestList.Select(s => s as IPendingSpecimenTestModel)
                //                                                 .Where(x => x != null);

                //foreach (var reviewedSpecimen in pendingSpecimenList)
                //{
                //    var existingRecord = PatientFullList.FirstOrDefault(x => x.PatientWithSpecimen.PatientSpecimenGuid == reviewedSpecimen.PatientSpecimenGuid);

                //    if (existingRecord != null)
                //    {
                //        for (int idx = 0; idx < existingRecord.PatientWithSpecimen.OrderableTestList.Count; idx++)
                //        {
                //            var reviewedOrderable = existingRecord.PatientWithSpecimen.OrderableTestList.FirstOrDefault(x => x.OrderableTestId == reviewedSpecimen.OrderableTest.OrderableTestId);

                //            if (reviewedOrderable != null)
                //            {
                //                //var tasTest = reviewedSpecimen as PendingSpecimenAboRhModel;
                //                //if (tasTest != null &&
                //                //    ((tasTest.PendingSpecimenAboRhModel != null && tasTest.PendingSpecimenAboRhModel.PendingTestStatus == PendingTestStatus.Pending) ||
                //                //    (tasTest.PendingSpecimenAbsModel != null && tasTest.PendingSpecimenAbsModel.PendingTestStatus == PendingTestStatus.Pending)))
                //                //{
                //                //    // This TAS still has a pending test associated with it so don't remove it
                //                //    continue;
                //                //}

                //                existingRecord.PatientWithSpecimen.OrderableTestList.Remove(reviewedOrderable);
                //                existingRecord.RaisePropertyChanged(existingRecord.OrderableTestNameListFormattedPropName);
                //                idx--;

                //                // No more tests exist for this specimen so remove it from the list
                //                if (!existingRecord.PatientWithSpecimen.OrderableTestList.Any())
                //                {
                //                    PatientList.Remove(existingRecord);
                //                    PatientFullList.Remove(existingRecord);
                //                }
                //            }
                //        }
                //    }
                //}
            }

            //if (!PatientFullList.Any())
            //{
            //    // No results exist any more so lets close the form
            //    VbecsBaseForm.Close();
            //}
        }

        /// <summary>
        /// Clear the search criteria and reset the results
        /// </summary>
        private void OnClearCommand()
        {
            SpecimenUidSearch = string.Empty;
            PatientSearch = string.Empty;
            PatientList = PatientFullList;
        }

        /// <summary>
        /// Search for a patient speciment by patient or specimenUid
        /// </summary>
        private void OnSearchCommand()
        {
            Guid? previousSelectedSpecimenUid = SelectedPatient == null ? null : (Guid?)SelectedPatient.PatientWithSpecimen.PatientSpecimenGuid;
            string patientName = "";
            string patientSSN = "";
            ParseSearchString(PatientSearch, out patientName, out patientSSN);

            if (!string.IsNullOrWhiteSpace(PatientSearch) &&
                string.IsNullOrWhiteSpace(patientName) &&
                string.IsNullOrWhiteSpace(patientSSN))
            {
                // User is searching for patient couldn't parse the search string
                // Thus not mathing results should return..
                PatientList.Clear();
            }
            else
            {
                // BR_3.20
                PatientList = PatientFullList.Where(x =>
                                       (string.IsNullOrWhiteSpace(patientName) || x.PatientWithSpecimen.Patient.LastName.IndexOf(patientName, StringComparison.InvariantCultureIgnoreCase) == 0)
                                    && (string.IsNullOrWhiteSpace(patientSSN) || x.PatientWithSpecimen.Patient.DisplayPatientID.Replace("-", "").IndexOf(patientSSN, StringComparison.InvariantCultureIgnoreCase) >= 0)
                                    && (string.IsNullOrWhiteSpace(SpecimenUidSearch) || x.PatientWithSpecimen.SpecimenUid.IndexOf(SpecimenUidSearch, StringComparison.InvariantCultureIgnoreCase) == 0)).ToObservableCollection();
            }

            SelectPatientSpecimen(previousSelectedSpecimenUid);
        }

        private void SelectPatientSpecimen(Guid? patientSpecimenGuid)
        {
            if (patientSpecimenGuid.HasValue)
            {
                foreach (var patient in PatientList)
                {
                    // Reset previously selected patient
                    if (patient.PatientWithSpecimen.PatientSpecimenGuid == patientSpecimenGuid)
                    {
                        // Gotta wait until the UI thread is available again to re-select
                        System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
                        {
                            SelectedPatient = patient;
                        }));
                    }
                }
            }
        }

        /// <summary>
        /// Parse the patient search string
        /// </summary>
        /// <param name="searchString"></param>
        /// <param name="patientName"></param>
        /// <param name="patientSSN"></param>
        private void ParseSearchString(string searchString, out string patientName, out string patientSSN)
        {
            patientName = string.Empty;
            patientSSN = string.Empty;

            if (!string.IsNullOrWhiteSpace(searchString))
            {
                Regex _lastfiveEx = RegularExpressions.PatientInitialAndLastFourSSN();
                Regex _ssnEx = RegularExpressions.PatientSsn();
                Regex _nameEx = RegularExpressions.PatientName();

                //Need to parse search string to determine if search is intended
                //to be on Name, full SSN, Last 4 or initial and Last 4.
                searchString = searchString.ToUpper();
                //
                //Check for Last 5
                if (_lastfiveEx.IsMatch(searchString))
                {
                    patientName = searchString.Substring(0, 1);
                    patientSSN = searchString.Substring(1, 4);
                    return;
                }
                //Check for SSN (full or last 4)
                if (_ssnEx.IsMatch(searchString))
                {
                    patientSSN = searchString;
                    return;
                }
                //Check for Name
                if (_nameEx.IsMatch(searchString))
                {
                    patientName = searchString;
                    return;
                }
            }
        }

        /// <summary>
        /// Open the window for the user to review the tests for acceptance or rejection
        /// </summary>
        private void OnOKCommand()
        {
            if (SelectedPatient != null)
            {
                var patientSpecimenGuid = SelectedPatient.PatientWithSpecimen.PatientSpecimenGuid;
                FrmWPFContainer openForm;
                if (_openSpecimenFormDictionary.TryGetValue(patientSpecimenGuid, out openForm))
                {
                    // Specimen is already open so bring it to front
                    openForm.BringToFront();
                }
                else
                {
                    // Create and show new instance of specimen window
                    var frm = ViewConstructor.CreateFormAndView<SpecimenTabsView, FrmWPFContainer>();
                    frm.MaximizeBox = true;
                    frm.MinimizeBox = true;
                    frm.WPFUserControl.DataContext = new SpecimenTabsViewModel(SelectedPatient.PatientWithSpecimen.Patient, patientSpecimenGuid, frm);
                    if (frm.DialogResult == System.Windows.Forms.DialogResult.Abort) return;

                    frm.MdiParent = VbecsBaseForm.MdiParent;
                    frm.FormClosed += frm_FormClosed;
                    _openSpecimenFormDictionary.Add(patientSpecimenGuid, frm);

                    frm.Show();
                }
            }
        }

        void frm_FormClosed(object sender, System.Windows.Forms.FormClosedEventArgs e)
        {
            // Remove form from out open form list
            var wpfForm = sender as FrmWPFContainer;
            if (wpfForm != null &&
                wpfForm.WPFUserControl != null)
            {
                var specimenTabsViewModel = wpfForm.WPFUserControl.DataContext as SpecimenTabsViewModel;

                if (specimenTabsViewModel != null &&
                    _openSpecimenFormDictionary.Keys.Any(x => x == specimenTabsViewModel.PatientSpecimenGuid))
                {
                    _openSpecimenFormDictionary.Remove(specimenTabsViewModel.PatientSpecimenGuid);
                }
            }
        }

        /// <summary>
        /// Close the window
        /// </summary>
        private void OnCancelCommand()
        {
            VbecsBaseForm.Close();
        }

        /// <summary>
        /// After the form is set we can construct the two tab view models and populate their data
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void PatientSelectionViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "VbecsBaseForm" &&
                VbecsBaseForm != null)
            {
                VbecsBaseForm.FormClosed += VbecsBaseForm_FormClosed;
                VbecsBaseForm.SetThisHelpFile(VbecsBaseForm, StrRes.HelpFiles.UC115.AutomatedTestingPatientSelectPendingOrders());
            }
        }
    }
}
